#include <math.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include "gtk/gtk.h"
#include "libgimp/gimp.h"
#include "libgimp/gimpui.h"
#include "effects.h"


#define PLUG_IN_NAME    "plug_in_gadget"
#define PLUG_IN_VERSION "0"


static char *button_names[] = 
{ "Rainbow", "Rubbing", "Mosaic", "Sniper", "Four" };
static char *parameter_names[] = 
{ "Zero", "One", "Two", "Three", "Four", "Five" };
#define ARRAY_SIZE(ar) (sizeof(ar) / sizeof(ar[0]))

struct gadget_param
{
  gint32  which;
  gdouble parameter[ARRAY_SIZE(parameter_names)];
};


static void query(void);
static void run(char *name, int nparams, GParam *param,
		int *nreturn_vals, GParam **return_vals);
static void gadget_go(gint32 d, struct gadget_param *p);
static gint gadget_dialog(struct gadget_param *p);
static void dialog_ok_callback(GtkWidget *widget, gpointer data);
static void dialog_cancel_callback(GtkWidget *widget, gpointer data);
static void dialog_close_callback(GtkWidget *widget, gpointer data);
static void dialog_button_callback(GtkWidget *widget, gpointer data);
static void dialog_entry_callback(GtkWidget *widget, gpointer data);

GPlugInInfo PLUG_IN_INFO =  { NULL, NULL, query, run };



int okay = 0;



/* We use the "main" provided for us */
MAIN();


/* The Gimp calls "query" to find-out more about us */
static void query(void)
{
  static GParamDef args[] = {
    { PARAM_INT32,    "run_mode",   "Interactive, non-interactive" },
    { PARAM_IMAGE,    "image",      "Input image" },
    { PARAM_DRAWABLE, "drawable",   "Input drawable" },
    { PARAM_INT32,    "which",      "Which effect to run" },
    { PARAM_FLOAT,    "parameter1",  "Parameter 1" },
    { PARAM_FLOAT,    "parameter2",  "Parameter 2" },
    { PARAM_FLOAT,    "parameter3",  "Parameter 3" },
    { PARAM_FLOAT,    "parameter4",  "Parameter 4" },
    { PARAM_FLOAT,    "parameter5",  "Parameter 5" },
  };

  gimp_install_procedure(PLUG_IN_NAME,
			 "A variety of filter effects",
			 "The Gadget plugin runs a variety of effects "
			 "created by the brilliant students of Washington "
			 "University's CS400-42, Computational Art.",

			 "Various authors",
			 "Various authors",
			 PLUG_IN_VERSION,
			 "<Image>/Gadget",
			 "RGB*, GRAY*",
			 PROC_PLUG_IN,
			 ARRAY_SIZE(args),
			 0,
			 args,
			 NULL);
}



/* The Gimp calls "run" when it's time for us to GO */
static void run(char *name, 
		int nparams, GParam *param,
		int *nreturn_vals, GParam **return_vals)
{
  GRunModeType run_mode;
  static struct gadget_param p;
  static GParam return_vals_storage[1];
  int i;

  *return_vals                    = return_vals_storage;
  *nreturn_vals                   = 1;
  (*return_vals)[0].type          = PARAM_STATUS;
  (*return_vals)[0].data.d_status = STATUS_EXECUTION_ERROR;

  run_mode = param[0].data.d_int32;


  p.which = 0;
  for (i = 0; i < ARRAY_SIZE(parameter_names); i++)
    p.parameter[i] = 0;


  switch (run_mode)
    {
    case RUN_INTERACTIVE:
      gimp_get_data(PLUG_IN_NAME, &p);
      if (!gadget_dialog(&p))
	return;
      gimp_set_data(PLUG_IN_NAME, &p, sizeof(p));

      break;

    case RUN_NONINTERACTIVE: 	  /* sorry - not what we're about */
      return;
	  
    case RUN_WITH_LAST_VALS:	  /* sorry - not what we do */
      gimp_get_data(PLUG_IN_NAME, &p);
      break;

    default:
      break;
    }

  // drawable = gimp_drawable_get(param[2].data.d_drawable);

  // gimp_tile_cache_ntiles((drawable->width + gimp_tile_width() - 1) /
  // gimp_tile_width());
			
  gadget_go(param[2].data.d_drawable, &p);

  
			
  /* If run mode is interactive, flush displays */
  if (run_mode != RUN_NONINTERACTIVE)
    gimp_displays_flush();
			

  // gimp_drawable_detach(drawable);
}



/* pop-up a dialog box and prompt the user for parameters */
static gint gadget_dialog(struct gadget_param *p)
{
  GtkWidget *dialog;
  GtkWidget *table;
  gint argc;
  gchar **argv;
  guchar *color_cube;
  int i;

  argc    = 1;
  argv    = g_new(gchar *, 1);
  argv[0] = g_strdup("gadget");

  gtk_init(&argc, &argv);
  
  gdk_set_use_xshm(gimp_use_xshm());
  
  gtk_preview_set_gamma(gimp_gamma());
  gtk_preview_set_install_cmap(gimp_install_cmap());
  color_cube = gimp_color_cube();
  gtk_preview_set_color_cube(color_cube[0], color_cube[1],
			     color_cube[2], color_cube[3]);
	
  gtk_widget_set_default_visual(gtk_preview_get_visual());
  gtk_widget_set_default_colormap(gtk_preview_get_cmap());

  /* Dialog box */

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), "Go Go Gadget");
  gtk_window_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  gtk_container_border_width(GTK_CONTAINER(dialog), 0);
  gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
		     (GtkSignalFunc) dialog_close_callback,
		     NULL);

  table = gtk_table_new(1, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     table, FALSE, FALSE, 0);
  gtk_widget_show(table);

  /* button area */
  {
    GtkWidget *button_table;
    GSList *group;

    button_table = gtk_table_new(1, ARRAY_SIZE(button_names), FALSE);
    gtk_container_border_width(GTK_CONTAINER(button_table), 6);
    gtk_table_attach(GTK_TABLE(table), button_table,
		     0, 1, 0, 1, 
		     GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 
		     0, 0);
    gtk_widget_show(button_table);

    group = NULL;

    for (i = 0; i < ARRAY_SIZE(button_names) ; i++)
      {
	GtkWidget *button;
	
	button = gtk_radio_button_new_with_label(group, button_names[i]);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
	gtk_table_attach(GTK_TABLE(button_table), button,
			 0, 1, i, i+1, 
			 GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 
			 0, 0);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   (GtkSignalFunc) dialog_button_callback,
			   (gpointer) &p->which);
	gtk_object_set_user_data(GTK_OBJECT(button), (gpointer)i);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), 
				    (i == p->which));
	gtk_widget_show(button);
      }

  }

  /* parameter area */
  {
    GtkWidget *parameter_table;
    
    parameter_table = gtk_table_new(2, ARRAY_SIZE(parameter_names),
				    FALSE);
    gtk_container_border_width(GTK_CONTAINER(parameter_table), 6);
    gtk_table_attach(GTK_TABLE(table), parameter_table,
		     0, 1, 1, 2, 
		     GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 
		     0, 0);
    gtk_widget_show(parameter_table);

    for (i = 0; i < ARRAY_SIZE(parameter_names); i++)
      {
	GtkWidget *label, *entry;
	char buf[100];

	label = gtk_label_new(parameter_names[i]);
	gtk_table_attach(GTK_TABLE(parameter_table), label,
			 0, 1, i, i+1, 
			 GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
			 0, 0);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	sprintf(buf,"%g", p->parameter[i]);
	gtk_entry_set_text(GTK_ENTRY(entry), buf);
	gtk_table_attach(GTK_TABLE(parameter_table), entry,
			 1, 2, i, i+1, 
			 GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
			 0, 0);
	gtk_signal_connect(GTK_OBJECT(entry), "changed",
                           (GtkSignalFunc) dialog_entry_callback,
                           &p->parameter[i]);
	gtk_widget_show(entry);
      }
    
  }

  /* OK button */
  {
    GtkWidget *ok_button;
    
    ok_button = gtk_button_new_with_label("OK");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		       ok_button, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
		       (GtkSignalFunc) dialog_ok_callback,
		       dialog);
    gtk_widget_show(ok_button);
    
  }

  gtk_widget_show(dialog);

  gtk_main();
  gdk_flush();

  return okay;

}

static void dialog_ok_callback(GtkWidget *widget, gpointer data)
{
  okay = 1;
  gtk_widget_destroy(GTK_WIDGET(data));
} 

static void dialog_cancel_callback(GtkWidget *widget, gpointer data)
{
  gtk_widget_destroy(GTK_WIDGET(data));
}

static void dialog_close_callback(GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
}

static void dialog_button_callback(GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON(widget)->active)
    *(gint32 *)data = (int)gtk_object_get_user_data(GTK_OBJECT(widget));
}

static void dialog_entry_callback(GtkWidget *widget, gpointer data)
{
  *(gdouble *)data = atof(gtk_entry_get_text(GTK_ENTRY(widget)));
}


static void gadget_go(gint32 d, struct gadget_param *p)
{
  GDrawable *drawable;
  GPixelRgn srcrgn, dstrgn;
  guchar *srcpix, *dstpix;
  int pixsize;


  drawable = gimp_drawable_get(d);

  /* create the regions */
  gimp_pixel_rgn_init (&srcrgn, drawable, 0, 0, 
		       drawable->width, drawable->height, FALSE, FALSE);
  gimp_pixel_rgn_init (&dstrgn, drawable, 0, 0, 
		       drawable->width, drawable->height, TRUE, TRUE);

  /* allocate the pixel buffers */
  pixsize = drawable->width * drawable->height * drawable->bpp;
  srcpix = (guchar*)malloc(pixsize);
  dstpix = (guchar*)malloc(pixsize);

  /* load the src pixel buffer */
  gimp_pixel_rgn_get_rect(&srcrgn, srcpix, 0, 0, 
			  drawable->width, drawable->height);


  /* do the effects! */
  switch(p->which)
    {
    case 0:
      rainbow(&p->parameter[0],
		  drawable->width, drawable->height, drawable->bpp,
		  srcpix, dstpix);
      break;

    case 1:
      rubbing(&p->parameter[0],
	   drawable->width, drawable->height, drawable->bpp,
	   srcpix, dstpix);
      break;
    case 2:
      mosaic(&p->parameter[0],
	     drawable->width, drawable->height, drawable->bpp,
	     srcpix, dstpix);
      break;
    case 3:
      sniper(&p->parameter[0],
	     drawable->width, drawable->height, drawable->bpp,
	     srcpix, dstpix);
      break;
    default: /* otherwise, make it black */
      bzero(dstpix, pixsize);
      break;
    }

  /* save the dst pixel buffer */
  gimp_pixel_rgn_set_rect(&dstrgn, dstpix, 0, 0,
			  drawable->width, drawable->height);

  free(srcpix);
  free(dstpix);

  gimp_drawable_flush(drawable);
  gimp_drawable_merge_shadow (drawable->id, TRUE);
  gimp_drawable_update(drawable->id, 0, 0, 
		       drawable->width, drawable->height);
  gimp_displays_flush();
  gimp_drawable_detach(drawable);
  
}
