#include "Object3d.h"
#include "Texture.h"
#include "../Options.h"
#include <math.h>

int Object3d::GLOBAL_OBJ_ID = 1;

float Object3d::cube_vertices_[] = {
	-1.0,-1.0,-1.0,
	1.0,-1.0,-1.0,
	1.0,1.0,-1.0, 
	-1.0,1.0,-1.0, 
	-1.0,-1.0,1.0, 
	1.0,-1.0,1.0, 
	1.0,1.0,1.0, 
	-1.0,1.0,1.0
};

float Object3d::cube_normals_[] = {
	-1.0,-1.0,-1.0,
	1.0,-1.0,-1.0,
	1.0,1.0,-1.0, 
	-1.0,1.0,-1.0, 
	-1.0,-1.0,1.0, 
	1.0,-1.0,1.0, 
	1.0,1.0,1.0, 
	-1.0,1.0,1.0
};

unsigned char Object3d::cube_indices_[] = {0,3,2,1,2,3,7,6,0,4,7,3,1,2,6,5,4,5,6,7,0,1,5,4};

void
Object3d::SetGlobalObjIDBase(int base) 
{
	Object3d::GLOBAL_OBJ_ID = base;
}

Object3d::Object3d() :
	objid_(GLOBAL_OBJ_ID++),
	vertices_(0),
	normals_(0),
	indices_(0),
	tex_vertices_(0),
	tex_indices_(0),
	bSelected_(false),
	texobj_(0),
	bHasTexture_(false),
  counter_(0)
{
	SetName("Undefined");
	ResetTransformations();
	SetColor(1, 1, 1);

	MATERIAL mat = {
		{0.0f,0.0f,0.0f},		/* ambient */
		{0.75f,0.75f,0.75f},	/* diffuse */
		{1.0f,1.0f,1.0f},		/* specular */
		{0.0f,0.0f,0.0f},		/* emission */
		1.0f					/* alpha */
	};
	materials_.push_back(mat);
}

Object3d::Object3d(float x, float y, float z) :
  Physics(x, y, z, 1.0),
	objid_(GLOBAL_OBJ_ID++),
	bSelected_(false),
  counter_(0)
{
	SetName("Undefined");
	ResetTransformations();
	SetColor(1, 1, 1);

	MATERIAL mat = {
		{0.0f,0.0f,0.0f},		/* ambient */
		{0.75f,0.75f,0.75f},	/* diffuse */
		{1.0f,1.0f,1.0f},		/* specular */
		{0.0f,0.0f,0.0f},		/* emission */
		1.0f					/* alpha */
	};
	materials_.push_back(mat);
}

Object3d& 
Object3d::operator=(Object3d &o)
{
	objid_ = o.objid_;
	SetName(o.GetName());
	for (int i = 0; i < 3; i++) {
		scale_[i] = o.scale_[i];
		color_[i] = o.color_[i];
	}
	
	SetVertices(o.vertices_, o.vertices_size_);
	SetNormals(o.normals_, o.normals_size_);
	SetIndices(o.indices_, o.indices_size_);
	SetTexVertices(o.tex_vertices_, o.tex_vertices_size_);
	SetTexIndices(o.tex_indices_, o.tex_indices_size_);
	materials_ = o.materials_;

	return *this;
}

Object3d::~Object3d() 
{

}

int 
Object3d::PreDraw(void)
{
#if 0
	glTranslatef(position_[X], position_[Y], position_[Z]);
	glScalef(scale_[X], scale_[Y], scale_[Z]);
	glRotatef(rotation_[X], 1.0f, 0.0, 0.0);
	glRotatef(rotation_[Y], 0.0, 1.0f, 0.0);
	glRotatef(rotation_[Z], 0.0, 0.0, 1.0f);
	glColor3f(color_[RED], color_[GREEN], color_[BLUE]);	
#endif

	//glScalef(scale_[X], scale_[Y], scale_[Z]);
	////glRotatef(rotation_[Z], 0.0, 0.0, 1.0f);
	////glRotatef(rotation_[Y], 0.0, 1.0f, 0.0);
	////glRotatef(rotation_[X], 1.0f, 0.0, 0.0);
	//glTranslatef(position_[X], position_[Y], position_[Z]);

	return ERR_OK;
}

int 
Object3d::Draw(void)
{
	if (glIsList(objid_) == GL_FALSE) {
		return ERR_FAIL;
	}

	glPushMatrix();

	glTranslatef(p_.x, p_.y, p_.z);
	glScalef(scale_[X], scale_[Y], scale_[Z]);
	glRotatef(r_.x, 1.0f, 0.0, 0.0);
	glRotatef(r_.y, 0.0, 1.0f, 0.0);
	glRotatef(r_.z, 0.0, 0.0, 1.0f);

	//
	// Draw selection indicator
	//
	if (bSelected_) {
  	glDisable(GL_TEXTURE_2D);  // stop using texture objects
	  glEnable(GL_POLYGON_OFFSET_FILL);
	  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	  glPolygonOffset(-1.0, -1.0);
	  glColor3f(0.0, 1.0, 0.0);
	  glCallList(objid_);
	  glDisable(GL_POLYGON_OFFSET_FILL);
	  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	}

	//
	// Turn on texturing
	//
	if (bHasTexture_) {
		glEnable(GL_TEXTURE_2D);
	}
	if (texobj_ != 0) {
		texobj_->Bind();
	}
	//
	// Draw object
	//
	glColor3f(color_[RED], color_[GREEN], color_[BLUE]);
	glCallList(objid_);
	
	glPopMatrix();



#if 0
	//
	// Draw selection box
	//
	if (bSelected_) {
		//float x1, x2, y1, y2, z1, z2;
		//if (GetBoundingBox(&x1, &x2, &y1, &y2, &z1, &z2) <= 0) {
		//	return ERR_FAIL;
		//}
		glEnableClientState(GL_NORMAL_ARRAY); 
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, cube_vertices_);
		glNormalPointer(GL_FLOAT,0, cube_normals_); 
		glPushMatrix();
		glTranslatef(position_[X], position_[Y], position_[Z]);
		glScalef(scale_[X] * 0.55, scale_[Y] * 0.55, scale_[Z] * 0.55);
		glRotatef(rotation_[X], 1.0f, 0.0, 0.0);
		glRotatef(rotation_[Y], 0.0, 1.0f, 0.0);
		glRotatef(rotation_[Z], 0.0, 0.0, 1.0f);
		glColor4f(0.0, 1.0, 0.0, 0.5);
		glDrawElements(GL_QUADS, 24, GL_UNSIGNED_BYTE, cube_indices_);
		glPopMatrix();
		glDisableClientState(GL_NORMAL_ARRAY); 
		glDisableClientState(GL_VERTEX_ARRAY);
	}
#endif

	return ERR_OK;
}

int 
Object3d::Update(double timedelta)
{
  Physics::Update(timedelta);
	return ERR_OK;
}


int
Object3d::ResetTransformations(void)
{
	//SetPos(0, 0, 0);
	SetScale(1, 1, 1);
	SetRotate(0, 0, 0);
	return ERR_OK;
}


int
Object3d::MakeUnitSize()
{
	float x1, x2, y1, y2, z1, z2;

	if (GetBoundingBox(&x1, &x2, &y1, &y2, &z1, &z2) <= 0) {
		return ERR_FAIL;
	}

	// Center object at (0, 0, 0)
	//SetTranslate((x1 + x2) / -2.0, (y1 + y2) / -2.0, (z1 + z2) / -2.0);
	SetPos((x1 + x2) / -2.0, (z1 + z2) / -2.0, (y1 + y2) / -2.0);

	// Scale object to (1, 1, 1) but maintain proportions
	float wx = x2 - x1;
	float wy = y2 - y1;
	float wz = z2 - z1;
	float scale;
	if (wx > wy)
		scale = wx;
	else
		scale = wy;
	if (wz > scale)
		scale = wz;
	scale = 1.0 / scale;
	SetScale(scale, scale, scale);

	return ERR_OK;
}

int 
Object3d::BeginDef(void)
{
	glNewList(objid_, GL_COMPILE);
	glPushMatrix();
	return ERR_OK;
}

int
Object3d::EndDef(void)
{
	glPopMatrix();
	glEndList();
	return ERR_OK;
}

void 
Object3d::ChangeScale(float dx, float dy, float dz)
{
	scale_[X] += dx;
	scale_[Y] += dy;
	scale_[Z] += dz;
}


float 
Object3d::Distance(Object3d &obj)
{
  float dx = p_.x - obj.p_.x;
  float dy = p_.y - obj.p_.y;
  float dz = p_.z - obj.p_.z;
  return sqrt((dx * dx + dy * dy + dz * dz));
}

void
Object3d::SetName(const char *name)
{
	strncpy(objname_, name, 64);
	objname_[63] = 0;  // guarantee null-terminated
}

void
Object3d::SetScale(float x, float y, float z)
{
	scale_[X] = x;
	scale_[Y] = y;
	scale_[Z] = z;
}

void
Object3d::SetColor(float r, float g, float b)
{
	color_[RED] = r;
	color_[GREEN] = g;
	color_[BLUE] = b;
}

void 
Object3d::SetMaterial(MATERIAL &mat)
{
	materials_.push_back(mat);
}

void
Object3d::SetSelected(bool pred)
{
	bSelected_ = pred;
	if (pred == true) {
		//SetColor(0.0, 1.0, 0.0);
	}
	else {
		//SetColor(1.0, 1.0, 1.0);
	}
}

void 
Object3d::SetTexture(Texture *texobj)
{
	texobj_ = texobj;
	bHasTexture_ = true;
}

void
Object3d::SetTexture(void)
{
	bHasTexture_ = true;
}

void 
Object3d::SetVertices(float *vertices, int size)
{
	if (vertices_ != 0) {
		delete vertices_;
	}
	vertices_ = vertices;
	vertices_size_ = size;
}

void 
Object3d::SetNormals(float *normals, int size)
{
	if (normals_ != 0) {
		delete normals_;
	}
	normals_ = normals;
	normals_size_ = size;
}

void 
Object3d::SetIndices(unsigned int *indices, int size)
{
	if (indices_ != 0) {
		delete indices_;
	}
	indices_ = indices;
	indices_size_ = size;
}

void 
Object3d::SetTexVertices(float *vertices, int size)
{
	if (tex_vertices_ != 0) {
		delete tex_vertices_;
	}
	tex_vertices_ = vertices;
	tex_vertices_size_ = size;
}

void 
Object3d::SetTexIndices(unsigned int *indices, int size)
{
	if (tex_indices_ != 0) {
		delete tex_indices_;
	}
	tex_indices_ = indices;
	tex_indices_size_ = size;
}

const char*
Object3d::GetName(void)
{
	return objname_;
}

int
Object3d::GetObjID(void)
{
	return objid_;
}

bool
Object3d::GetSelected(void)
{
	return bSelected_;
}

float*
Object3d::GetScale(void)
{
	return scale_;
}

float*
Object3d::GetColor(void)
{
	return color_;
}

int
Object3d::GetBoundingBox(float *x1, float *x2, float *y1, float *y2, float *z1, float *z2)
{
	if ((vertices_size_ == 0) || (indices_size_ == 0)) {
		return ERR_FAIL;
	}

	float x, y, z;
	*x1 = *x2 = vertices_[0];
	*y1 = *y2 = vertices_[1];
	*z1 = *z2 = vertices_[2];
	for (int i = 0; i < indices_size_; i++) {
		x = vertices_[indices_[i] * 3];
		y = vertices_[indices_[i] * 3 + 1];
		z = vertices_[indices_[i] * 3 + 2];
		if (x < *x1) 
			*x1 = x;
		else if (x > *x2) 
			*x2 = x;
		if (y < *y1) 
			*y1 = y;
		else if (y > *y2) 
			*y2 = y;
		if (z < *z1) 
			*z1 = z;
		else if (z > *z2) 
			*z2 = z;
	}
	
	return ERR_OK;
}



void 
Object3d::SetMaterial(int mode, float *f, float alpha)
{
	GLfloat d[4];
	d[0]=f[0];
	d[1]=f[1];
	d[2]=f[2];
	d[3]=alpha;
	glMaterialfv(GL_FRONT_AND_BACK, mode, d);
}

void 
Object3d::SelectMaterial(int i)
{
	if (i >= materials_.size()) {
		return;
	}
	GLfloat alpha = materials_[i].alpha;
	SetMaterial(GL_AMBIENT, materials_[i].ambient, alpha);
	SetMaterial(GL_DIFFUSE, materials_[i].diffuse, alpha);
	SetMaterial(GL_SPECULAR, materials_[i].specular, alpha);
	SetMaterial(GL_EMISSION, materials_[i].emission, alpha);
}

