#include "Vertex.h"
#include <stdio.h>

Vertex::Vertex() :
	mass_(1),
  counter_(0),
  period_(0),
  path_idx_(-1),
  path_type_(PATH_SINGLE)
{
  //fprintf(stderr, "Vertex()\n");
}

Vertex::Vertex(const Vertex &l) :
  p_(l.p_),
  r_(l.r_),
  s_(l.s_),
  c_(l.c_),
  dp_(l.dp_),
  dr_(l.dr_),
  ds_(l.ds_),
  dc_(l.dc_),
  velp_(l.velp_),
  velr_(l.velr_),
  vels_(l.vels_),
  velc_(l.velc_),
  accelp_(l.accelp_),
  accelr_(l.accelr_),
  accels_(l.accels_),
  accelc_(l.accelc_),
  restp_(l.restp_),
  restr_(l.restr_),
  rests_(l.rests_),
  restc_(l.restc_),
  mass_(l.mass_),
  counter_(l.counter_),
  period_(l.period_),
  path_idx_(l.path_idx_),
  path_type_(l.path_type_)
{
  //fprintf(stderr, "Vertex(copy)\n");
}

Vertex::Vertex(float x, float y, float z) :
  p_(x, y, z),
  restp_(x, y, z),
  mass_(1),
  counter_(0),
  period_(0),
  path_idx_(-1),
  path_type_(PATH_SINGLE)
{
}
  
Vertex::Vertex(float x, float y, float z, float mass) :
  p_(x, y, z),
  restp_(x, y, z),
  mass_(mass),
  counter_(0),
  period_(0),
  path_idx_(-1),
  path_type_(PATH_SINGLE)
{
}

Vertex::~Vertex()
{
  for (int i = 0; i < path_.size(); i++) {
    if (path_[i].v != this) delete path_[i].v;
  }
  path_.clear();
}

void
Vertex::Reset(void)
{
  p_ = restp_;
  r_ = restr_;
  s_ = rests_;
  c_ = restc_;
  dp_ = 0;
  dr_ = 0;
  ds_ = 0;
  dc_ = 0;
  velp_ = 0;
  velr_ = 0;
  vels_ = 0;
  velc_ = 0;
  accelp_ = 0;
  accelr_ = 0;
  accels_ = 0;
  accelc_ = 0;
  counter_ = 0;
  period_ = 0;
}

void
Vertex::Update(float timedelta)
{
  /* 
  * Is there a waypoint to go to? 
  * If so, update waypoint stuff.
  */
  if (path_idx_ < path_.size()) {
    /*
    * Ease
    */
    if (path_[path_idx_].type == WAYPOINT_EASE) {
      float t = period_ * EASE_THRESHOLD;
      if (counter_ < t && counter_ + timedelta >= t) {  /* 0...THRESHOLD */
        //fprintf(stderr, "ease threshold %f\n", t);
        accelp_ = (accelp_ * -counter_) / (period_ - counter_);
        accelr_ = (accelr_ * -counter_) / (period_ - counter_);
        accels_ = (accels_ * -counter_) / (period_ - counter_);
        accelc_ = (accelc_ * -counter_) / (period_ - counter_);
      }
    }
    /*
    * Linear
    */
    else if (path_[path_idx_].type == WAYPOINT_LINEAR) {
      
    }

    /*
    * About to expire for good, i.e. reach the waypoint? 
    * Stop acceleration, "jump" to position. 
    * Go to next waypoint.
    */
    if (counter_ + timedelta >= period_) {
#if 0
      accelp_ = 0;
      accelr_ = 0;
      accels_ = 0;
      accelc_ = 0;
      velp_ = 0;
      velr_ = 0;
      vels_ = 0;
      velc_ = 0;
      //printf("p_=(%f, %f, %f) path_idx_=%d\n", p_.x, p_.y, p_.z, path_idx_);
      p_ = path_[path_idx_].v->p_;
      r_ = path_[path_idx_].v->r_;
      s_ = path_[path_idx_].v->s_;
      c_ = path_[path_idx_].v->c_;
#endif

      if (path_idx_ + 1 < path_.size()) {
        GotoWaypoint(path_idx_ + 1);
      }
      else if (path_type_ == PATH_LOOP) {
        GotoWaypoint(0);
      }
    }
  }

  /* 
   * Particle acceleration
   */
  velp_ = velp_ + accelp_ * timedelta;
  velr_ = velr_ + accelr_ * timedelta;
  vels_ = vels_ + accels_ * timedelta;
  velc_ = velc_ + accelc_ * timedelta;

  /* Reducing acceleration?  Seems kinda weird.  Leave it for now.  */
  //accelp_ = accelp_ - accelp_ * timedelta;

	/* Update position */
  dp_ = velp_ * timedelta;
  dr_ = velr_ * timedelta;
  ds_ = vels_ * timedelta;
  dc_ = velc_ * timedelta;
  p_ = p_ + dp_;
  r_ = r_ + dr_;
  s_ = s_ + ds_;
  c_ = c_ + dc_;

  /* Update timer */
  counter_ += timedelta;
}

void
Vertex::ClearPath(void)
{
  for (int i = 0; i < path_.size(); i++) {
    if (path_[i].v != this) delete path_[i].v;
  }
  path_.clear();
  path_idx_ = -1;
}

Vertex*
Vertex::Goto(float time, Vertex &v, int type)
{
  if (time == 0) {
    Reset();
    p_ = v.p_;
    return this;
  }
  else {
    path_type_ = PATH_LOOP;

    WAYPOINT wp;
    wp.v = new Vertex(v);
    wp.time = time;
    wp.type = type;
    path_.push_back(wp);
    /* If this is the first waypoint we've added, then set up motion/timer stuff */
    if (path_.size() == 1) {
      GotoWaypoint(0);
    }
    return wp.v;
  }
}

Vertex*
Vertex::Goto(float time, float x, float y, float z, int type)
{
  if (time == 0) {
    Reset();
    p_.Set(x, y, z);
    return this;
  }
  else {
    path_type_ = PATH_LOOP;

    WAYPOINT wp;
    wp.v = new Vertex(*this);
    wp.v->p_.Set(x, y, z);
    wp.time = time;
    wp.type = type;
    path_.push_back(wp);
    /* If this is the first waypoint we've added, then set up motion/timer stuff */
    if (path_.size() == 1) {
      GotoWaypoint(0);
    }
    return wp.v;
  }
}

Vertex*
Vertex::Goto(float time, vector3f pos, int type)
{
  return Goto(time, pos.x, pos.y, pos.z, type);
}

Vertex*
Vertex::GotoRest(float time)
{
  return Goto(time, restp_.x, restp_.y, restp_.z, WAYPOINT_EASE);
}

void 
Vertex::GotoWaypoint(int path_idx)
{
  path_idx_ = path_idx;
  WAYPOINT &wp = path_[path_idx];
  counter_ = 0;
  period_ = wp.time;
  if (wp.time == 0) {
    p_ = wp.v->p_;
    r_ = wp.v->r_;
    s_ = wp.v->s_;
    c_ = wp.v->c_;
  }
  else if (wp.type == WAYPOINT_EASE) {
    float t = wp.time * EASE_THRESHOLD;
    accelp_ = (wp.v->p_ - p_) * 2 / (t * wp.time);  /* see notes in Intel notebook */
    accelr_ = (wp.v->r_ - r_) * 2 / (t * wp.time); 
    accels_ = (wp.v->s_ - s_) * 2 / (t * wp.time); 
    accelc_ = (wp.v->c_ - c_) * 2 / (t * wp.time); 
    velp_ = 0;
    velr_ = 0;
    vels_ = 0;
    velc_ = 0;
  }
  else if (wp.type == WAYPOINT_LINEAR) {
    velp_ = (wp.v->p_ - p_) / wp.time;
    velr_ = (wp.v->r_ - r_) / wp.time;
    vels_ = (wp.v->s_ - s_) / wp.time;
    velc_ = (wp.v->c_ - c_) / wp.time;
    accelp_ = 0;
    accelr_ = 0;
    accels_ = 0;
    accelc_ = 0;
  }
}