#include "globals.h"
#include "3d/Object3d.h"
#include "3d/Texture.h"
#include "FileLoader.h"
#include <math.h>  // fabs

#ifdef WIN32
#include <windows.h>  // QueryPerformanceCounter...
#endif

Options* Options::instance_ = 0;

Options* Options::instance(void) 
{
	if (instance_ == 0)
		instance_ = new Options();
	return instance_;
}


//GLfloat global_ambient[] = {0.5f, 0.5f, 0.5f, 1.0f};		/* World ambient light. */
//GLfloat light_ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};			/* Light ambient light. */
//GLfloat light_diffuse[] = {1.0f, 1.0f, 1.0f, 1.0f};			/* White diffuse light. */
//GLfloat light_specular[] = {1.0f, 1.0f, 1.0f, 1.0f};		/* Color of light. */
//GLfloat light_position[] = {-100.0f, 100.0f, 100.0f, 0.0f};		/* Directional light */


Options::Options(void) :
	bLightingGlobal_(1),
	bLightingLight01_(1),
	bColorMaterial_(1),
	num_models_(0),
	curModel_(-1),
	num_textures_(0),
	curTexture_(0)
{

	//
	// These settings really should either be saved in the
	// registry, or (even better) saved with the map
	//
	global_ambient_[0] = 0.2f; 
	global_ambient_[1] = 0.2f;
	global_ambient_[2] = 0.2f;
	global_ambient_[3] = 1.0f;
	light_ambient_[0] = 0.0f;
	light_ambient_[1] = 0.0f;
	light_ambient_[2] = 0.0f;
	light_ambient_[3] = 1.0f;
	light_diffuse_[0] = 1.0f;
	light_diffuse_[1] = 1.0f;
	light_diffuse_[2] = 1.0f;
	light_diffuse_[3] = 1.0f;
	light_specular_[0] = 1.0f;
	light_specular_[1] = 1.0f;
	light_specular_[2] = 1.0f;
	light_specular_[3] = 1.0f;
	light_position_[0] = 320.0f;
	light_position_[1] = 240.0f;
	light_position_[2] = 200.0f;
	light_position_[3] = 1.0f;  // directional light(=0?)

#ifdef WIN32
	if (!QueryPerformanceFrequency((LARGE_INTEGER *)&timer_freq_)) {
		return;
	}
	timer_freqINV_ =  (double)(1 / (double)(timer_freq_));
#endif

	models_.reserve(64);
	textures_.reserve(64);
}

Options::~Options(void) 
{
}


/**********************************************************
 *
 * Object_Callback()
 *
 * Processes each model as it is loaded from a .ase file
 *
 **********************************************************/

int
Object_Callback(Object3d &o)
{

	return 1;  // push onto list
}


double
Options::GetTimeDelta(void)
{
#ifdef WIN32
	__int64 t;
  double delta;
	QueryPerformanceCounter((LARGE_INTEGER *)&t);
  delta = (t - prev_time_) * timer_freqINV_;
  prev_time_ = t;
	return delta;
#else
  int cur_time;
  double delta;
  struct timeval tv;
  struct timezone tz;
  gettimeofday(&tv, &tz);
  cur_time = tv.tv_sec * 1000000 + tv.tv_usec;
  delta = (double)(cur_time - prev_time_) / 1000000.0;
  prev_time_ = cur_time;
  return delta;
#endif
}

double
Options::GetTime(void)
{
#ifdef WIN32
	__int64 t;
	QueryPerformanceCounter((LARGE_INTEGER *)&t);
  return t * timer_freqINV_;
#else
  struct timeval tv;
  struct timezone tz;
  gettimeofday(&tv, &tz);
  return ((double)tv.tv_sec + (double)tv.tv_usec / 1000000.0);
#endif
}

int
Options::FindTexture(const char *name) 
{
	for (int i = 0; i < textures_.size(); i++) {
		if (strcmp(name, textures_[i]->GetName()) == 0) {
			return i;
		}
	}
	return -1;
}


///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////


void
multMatrices(const GLfloat a[16], const GLfloat b[16], GLfloat r[16])
{
	int i, j;
	
	for (i = 0; i < 4; i++) {
		for (j = 0; j < 4; j++) {
			r[i * 4 + j] =
				a[i * 4 + 0] * b[0 * 4 + j] +
				a[i * 4 + 1] * b[1 * 4 + j] +
				a[i * 4 + 2] * b[2 * 4 + j] +
				a[i * 4 + 3] * b[3 * 4 + j];
		}
	}
}

void
makeIdentity(GLfloat m[16])
{
	m[0 + 4 * 0] = 1;
	m[0 + 4 * 1] = 0;
	m[0 + 4 * 2] = 0;
	m[0 + 4 * 3] = 0;
	m[1 + 4 * 0] = 0;
	m[1 + 4 * 1] = 1;
	m[1 + 4 * 2] = 0;
	m[1 + 4 * 3] = 0;
	m[2 + 4 * 0] = 0;
	m[2 + 4 * 1] = 0;
	m[2 + 4 * 2] = 1;
	m[2 + 4 * 3] = 0;
	m[3 + 4 * 0] = 0;
	m[3 + 4 * 1] = 0;
	m[3 + 4 * 2] = 0;
	m[3 + 4 * 3] = 1;
}

/*
** inverse = invert(src)
*/
int
invertMatrix(const GLfloat src[16], GLfloat inverse[16])
{
	int i, j, k, swap;
	double t;
	GLfloat temp[4][4];
	
	for (i = 0; i < 4; i++) {
		for (j = 0; j < 4; j++) {
			temp[i][j] = src[i * 4 + j];
		}
	}
	makeIdentity(inverse);
	
	for (i = 0; i < 4; i++) {
    /* 
		** Look for largest element in column */
		swap = i;
		for (j = i + 1; j < 4; j++) {
			if (fabs(temp[j][i]) > fabs(temp[i][i])) {
				swap = j;
			}
		}
		
		if (swap != i) {
		/* 
			** Swap rows. */
			for (k = 0; k < 4; k++) {
				t = temp[i][k];
				temp[i][k] = temp[swap][k];
				temp[swap][k] = t;
				
				t = inverse[i * 4 + k];
				inverse[i * 4 + k] = inverse[swap * 4 + k];
				inverse[swap * 4 + k] = t;
			}
		}
		if (temp[i][i] == 0) {
		/* 
		** No non-zero pivot.  The matrix is singular, which
		shouldn't ** happen.  This means the user gave us a
			bad matrix. */
			return 0;
		}
		t = temp[i][i];
		for (k = 0; k < 4; k++) {
			temp[i][k] /= t;
			inverse[i * 4 + k] /= t;
		}
		for (j = 0; j < 4; j++) {
			if (j != i) {
				t = temp[j][i];
				for (k = 0; k < 4; k++) {
					temp[j][k] -= temp[i][k] * t;
					inverse[j * 4 + k] -= inverse[i * 4 + k] * t;
				}
			}
		}
	}
	return 1;
}
