#include "UIMap.h"
#include "globals.h"
#include "fileloader.h"
#include "3d/Object3d.h"
#include "3d/Texture.h"
#include "3d/Square.h"
#include "3d/Circle.h"
#include "3d/Triangle.h"
#include "3d/Parallelogram.h"
#include "3d/Mesh.h"
#include <math.h>

float ml_ambient_[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
float ml_diffuse_[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
float ml_specular_[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
float ml_pos_[4] = { 0.0f, 0.0f, 5.0f, 1.0f };


UIMap::UIMap() :
	groundplane_(new Object3d()),
  width_(640),
  height_(480),
	camera_x_(0),
	camera_y_(0),
	camera_z_(600),  // 772?
	near_(10),
	far_(5000),
	fov_(45),
	mouse_scalex_(1),
	mouse_scaley_(1),
  mouse_x_down_(-1),
  mouse_y_down_(-1),
  marq_type_(MARQ_NONE),
  effect_type_(EFFECT_NONE),
  leftstate_(STATE_MANIPULATE),
  rightstate_(STATE_NONE),
  obj_undercurs_(0),
  counter_(0),
  period_(0)
{
	obj_vec_.reserve(100);
	obj_selected_.reserve(100);

}

UIMap::~UIMap()
{
  delete groundplane_;
}

int
UIMap::Initialize(const char *basefn)
{
	Options &o = *(Options::instance());

	//
	// RESET MAP STATE
	//
	obj_vec_.clear();
	obj_selected_.clear();
	

  char str_img[60];
  char str_mask[60];
  char str_map[60];
  sprintf(str_img, "%s.ppm", basefn);
  sprintf(str_mask, "%s_mask.pgm", basefn);
  sprintf(str_map, "%s.map", basefn);

  /* Load main image */
  int bitdepth;
  img_ = load_ppm32(str_img, &img_width_, &img_height_, &bitdepth);
  
  //unsigned char *img_ = load_bmp("stonetile.bmp", &width_, &height_);
  //bitdepth = 32;

  printf("%s: (%d, %d) x %d\n", str_img, width_, height_, bitdepth);
	Texture *texobj = new Texture(basefn);
  texobj->Initialize(img_width_, img_height_);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, img_width_, img_height_, 0, GL_RGBA, GL_UNSIGNED_BYTE, img_);
	o.textures_.push_back(texobj);
  o.num_textures_++;

  /* Create groundplane */
  width_ = 640;
  height_ = 480;
  groundplane_->SetTexture(texobj);
  groundplane_->SetScale(1, 1, 1);
  groundplane_->SetPos(0, 0, 0);
  //groundplane_->SetRotate(180, 0, 0
  groundplane_->SetColor(1.0, 1.0, 1.0);
  groundplane_->BeginDef();
  glBegin(GL_QUADS);
    glTexCoord3f(0, 0, 0);
    glVertex3f(0, 0, 0);  // lower left
    glTexCoord3f(0, 1, 0);
    glVertex3f(0, height_, 0);  // upper left
    glTexCoord3f(1, 1, 0);
    glVertex3f(width_, height_, 0);  // upper right
		glTexCoord3f(1, 0, 0);
		glVertex3f(width_, 0, 0);  // lower right
	glEnd();
	groundplane_->EndDef();

  /* Load grayscale mask image */
  rgnmap_.mask = load_pgm(str_mask, &rgnmap_.width, &rgnmap_.height);
  printf("%s: (%d, %d) x 8\n", str_mask, rgnmap_.width, rgnmap_.height);

  /* Load region map information */
  load_map(str_map, rgnmap_);
  printf("%s: OK\n", str_map);

  /* Build button shapes */
  //BuildShapes();
  BuildUI();

  SetMouseScale();
  mouse_scalex_ *= 640;
  mouse_scaley_ *= 480;

	/* Center camera in middle of map */
	camera_x_ = width_ / 2.0;
	camera_y_ = height_ / 2.0;

  /* Delete texture bitmaps now that they are compiled into the display list */
  //delete [] img_;

	/* Enable the mouse light */
	glLightfv(GL_LIGHT1, GL_AMBIENT, ml_ambient_);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, ml_diffuse_);
	glLightfv(GL_LIGHT1, GL_SPECULAR, ml_specular_);
	glLightfv(GL_LIGHT1, GL_POSITION, ml_pos_);
	glEnable(GL_LIGHT1);
  //glDisable(GL_LIGHT1);


  return ERR_OK;
}

void
UIMap::BuildShapes(void) 
{
  Mesh *shape;
  int cellwidth = width_ / rgnmap_.width;
  int cellheight = height_ / rgnmap_.height;
  int cellwidth2 = cellwidth / 2;
  int cellheight2 = cellheight / 2;
  obj_vec_.reserve(rgnmap_.width * rgnmap_.height);
  for (int j = 0; j < rgnmap_.height; j++) {
    for (int i = 0; i < rgnmap_.width; i++) {
      shape = CreateShape(rgnmap_.map[(int)rgnmap_.mask[j * rgnmap_.width + i]].region, i * cellwidth + cellwidth2, j * cellheight + cellheight2);
    }
  }
}

void
UIMap::BuildUI(void) 
{
  buttons_.resize(4 * 2);

  buttons_[0] = new Square(20, 10, 4, 40);
  buttons_[1] = new Parallelogram(20, 10, 5, 34);
  
  buttons_[2] = new Square(80, 10, 4, 40);
  buttons_[3] = new Circle(80, 10, 5, 17);
  
  buttons_[4] = new Square(140, 10, 4, 40);
  buttons_[5] = new Square(140, 10, 5, 28);
  
  buttons_[6] = new Square(200, 10, 4, 40);
  buttons_[7] = new Triangle(200, 10, 5, 34, 34);
  for (int i = 0; i < buttons_.size(); i++) {
    if ((i % 2) == 0) buttons_[i]->SetColor(0, 0, 0.7);
    else buttons_[i]->SetColor(0.9, 0.9, 0.0);
  }
}

void
UIMap::SetUI(int type)
{
  for (int i = 0; i < buttons_.size(); i++) {
    buttons_[i]->SetSelected(false);
  }
  buttons_selected_.clear();
  curshape_ = type;
  buttons_[(curshape_ << 1)]->SetSelected(true);
  buttons_[(curshape_ << 1) + 1]->SetSelected(true);
}

void 
UIMap::SetProjection(int x, int y, int width, int height)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(x, y, width, height);
	gluPerspective(fov_, (GLfloat)(width) / (GLfloat)(height), near_, far_);
  //glRotatef(180, 1.0, 0, 0);
	glTranslatef(-camera_x_, -camera_y_, -camera_z_);
}

int
UIMap::Draw(int mode)
{
  int i;
	Options &o = *(Options::instance());

	if (mode == GL_RENDER) {
		glEnable(GL_DEPTH_TEST); /* Enable hidden--surface--removal */
		glEnable(GL_LIGHTING);
		if (o.bLightingLight01_) glEnable(GL_LIGHT0);
		if (o.bColorMaterial_) glEnable(GL_COLOR_MATERIAL);
	}
	else {
		glDisable(GL_LIGHTING);
	}

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

  /* Reload texture. Don't do this unless changes have been made. */
  //glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, img_width_, img_height_, GL_RGBA, GL_UNSIGNED_BYTE, img_);

  /*
	 * Draw groundplane
	 */
  glDisable(GL_LIGHTING);
	if (mode == GL_RENDER) {
		if (FAIL(groundplane_->Draw())) {
			return ERR_FAIL;
		}
	}
	glEnable(GL_LIGHTING);

	/*
	 * Draw all objects stored in vector
	 */
	for (i = 0; i < obj_vec_.size(); i++) {
		if (mode == GL_SELECT) glLoadName(i);
		obj_vec_[i]->Draw();
	}

  /*
   * Draw UI buttons
   */
  glDisable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);  // stop using texture objects
	glLineWidth(8.0);
  if (mode == GL_RENDER) {
    glBegin(GL_QUADS);
      glColor3f(0.9, 0.9, 0.9);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-10.0, 20.0, 3.0);

      glColor3f(0.9, 0.9, 0.9);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(230.0, 20.0, 3.0);

      glColor3f(0.3, 0.3, 0.3);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(230.0, 0.0, 3.0);
    
      glColor3f(0.3, 0.3, 0.3);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-10.0, 0.0, 3.0);
    glEnd();
  }
  glLineWidth(3.0);
  for (i = 0; i < buttons_.size(); i++) {
    if (mode == GL_SELECT) glLoadName(i + obj_vec_.size());
    buttons_[i]->Draw();
  }
  glEnable(GL_POLYGON_OFFSET_FILL);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glPolygonOffset(-1.0, -1.0);
  glColor3f(1.0, 1.0, 1.0);
  for (i = 0; i < buttons_.size(); i++) {
    //buttons_[i]->Draw();
  }
  glDisable(GL_POLYGON_OFFSET_FILL);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glEnable(GL_LIGHTING);

  /*
   * Draw some kind of selection indicator (marq)?
   */
	if (marq_type_ != MARQ_NONE && mode == GL_RENDER) {
		glDisable(GL_DEPTH_TEST); 
 		glDisable(GL_COLOR_MATERIAL);
		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_2D);

		GLint viewport[4];
		glGetIntegerv(GL_VIEWPORT, viewport);
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		gluOrtho2D(0, viewport[2], viewport[3], 0);
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		glColor3f(1.0, 1.0, 1.0);

    if (marq_type_ == MARQ_LINE || marq_type_ == MARQ_CIRC || marq_type_ == MARQ_TRI1) {
  		glBegin(GL_LINES);
			  glVertex2f(marq_x_, marq_y_);
			  glVertex2f(mouse_x_, mouse_y_);
		  glEnd();
    }
    else if (marq_type_ == MARQ_TRI2) {
  		glBegin(GL_LINES);
        glVertex2f(marq_x_, marq_y_);
        glVertex2f(marq_x2_, marq_y2_);

        glVertex2f(marq_x2_, marq_y2_);
			  glVertex2f(mouse_x_, mouse_y_);
			  
        glVertex2f(mouse_x_, mouse_y_);
        glVertex2f(marq_x_, marq_y_);
		  glEnd();
    }
		glPopMatrix();
		glMatrixMode(GL_PROJECTION);
		glPopMatrix();
	}
	return ERR_OK;
}

void
UIMap::Update(float timedelta)
{
  int i;
  Options &o = *Options::instance();
  double time = o.GetTime();
  Mesh *s;
  int cellwidth = width_ / rgnmap_.width;
  int cellheight = height_ / rgnmap_.height;

  /* Check inactivity timer */
  inactivity_ += timedelta;
  if (inactivity_ > RESET_TIME - 3.0) {
    ShootUp();
  }
  else if (inactivity_ > RESET_TIME) {
    for (i = 0; i < obj_vec_.size(); i++) {
      delete obj_vec_[i];
    }
    obj_vec_.clear();
    obj_selected_.clear();
    obj_undercurs_ = 0;
  }

  /* If user holds down right button, make object grow */
  if (obj_undercurs_ && rightstate_ == STATE_PLACE) {
    obj_undercurs_->ChangeScale(1.0 + 1.3 * timedelta, 1.0 + 1.3 * timedelta, 1);
  }

  for (i = 0; i < obj_vec_.size(); i++) {
    s = obj_vec_[i];
    s->Update(timedelta);
#if 0
    /* lifespan over? */
    if (s->timestamp_ + 30.0 < time) {
      s->ChangePos(0, 0, 1000);
      s->ClearPath();
    }
#endif
  }

  /* Make local copy */
  int effect_type = effect_type_;

  /* About to expire? */
  if (period_ > 0 && counter_ + timedelta > period_) {
    switch (effect_type) {
    case EFFECT_SHOOTDOWN:
      o.light_position_[2] -= 400;
  	  glLightfv(GL_LIGHT0, GL_POSITION, o.light_position_);
      break;
    }
    effect_type_ = EFFECT_NONE;
    counter_ = period_;
  }
  /* All OK */
  else {
    counter_ += timedelta;
  }

#if 0
  switch (effect_type) {
  case EFFECT_NONE:
    /* This shouldn't occur - why set timer for EFFECT_NONE? */
    break;
  case EFFECT_SHOOTUP:
  	for (i = 0; i < obj_vec_.size(); i++) {
      obj_vec_[i]->Move(0, 0, 100 * counter_ / period_);
    }
    break;
  case EFFECT_SHOOTDOWN:
  	for (i = 0; i < obj_vec_.size(); i++) {
      obj_vec_[i]->Move(0, 0, -100 * counter_ / period_);
    }
    break;
  }
#endif
}

void
UIMap::MouseDown(int x, int y, int btn)
{
  vector3f m;
  Mesh *s1, *s2, *s3;
  Vertex *v1, *v2, *v3;
  float dt1, dt2;
  float time = Options::instance()->GetTime();;

  mouse_x_ = x;
  mouse_y_ = y;
  mouse_x_down_ = x;
  mouse_y_down_ = y;
  PickObject(x, y);
  
  if (buttons_selected_.size() > 0) {
    SetUI((buttons_selected_[0]) >> 1);
    return;
  }
    
  if (btn == GLUT_RIGHT_BUTTON) {
    if (obj_selected_.size() > 0) return;
    rightstate_ = STATE_PLACE;
    s1 = CreateShape(curshape_, x, height_ - y);
    obj_undercurs_ = s1;
  }
  else if (btn == GLUT_LEFT_BUTTON) {
    if (obj_selected_.size() >= 1) s1 = obj_vec_[obj_selected_[0]];
    if (obj_selected_.size() >= 2) s2 = obj_vec_[obj_selected_[1]];
    if (obj_selected_.size() >= 3) s3 = obj_vec_[obj_selected_[2]];
    /*
     * Object here?
     */
    if (obj_selected_.size() > 0) {
      switch (marq_type_) {
      case MARQ_NONE:
        marq_x_ = x;
			  marq_y_ = y;
        marq_time_ = time;
        if (strcmp(s1->GetName(), "triangle") == 0) {
          marq_type_ = MARQ_TRI1;
        }
        else if (strcmp(s1->GetName(), "circle") == 0) {
          marq_type_ = MARQ_CIRC;
        }
        else {
          marq_type_ = MARQ_LINE;
        }
        break;
      case MARQ_LINE:
        if (obj_selected_.size() < 2)
          return;
        dt1 = time - marq_time_;
        if (dt1 < 8.0) dt1 *= 2;
        s1->ClearPath();
        s2->ClearPath();
        /* Same type of shape? */
        if (strcmp(s1->GetName(), s2->GetName()) == 0) {
          const float ratio = 1.2;  // hack
          v1 = s1->Goto(dt1, *s1);
          v1->s_ = s2->s_;
          s1->GotoWaypoint(0);
          
          v2 = s2->Goto(dt1, *s2);
          v2->s_ = s1->s_;
          s2->GotoWaypoint(0);
        }
        /* Different type of shape */
        else {
          if (strcmp(s1->GetName(), "square") == 0) {
            v1 = s1->Goto(dt1, *s2);
            v1->r_ = 0;
            s1->GotoWaypoint(0);
            v2 = s2->Goto(dt1, *s1);
            v2->r_ = 0;
            s2->GotoWaypoint(0);
          }
          else {
            v1 = s1->Goto(dt1, *s2);
            v1->r_ = 0;
            v2 = s2->Goto(dt1, *s1);
            v2->r_ = 0;
          }  
        }
        printf("connection (time=%f)\n", (float)(time - marq_time_));
        marq_type_ = MARQ_NONE;
        DeselectAll();
        break;
      case MARQ_CIRC:
        if (obj_selected_.size() < 2)
          return;
        dt1 = (time - marq_time_);
        s1->ClearPath();
        s2->ClearPath();
        /* Same type of shape? */
        if (strcmp(s1->GetName(), s2->GetName()) == 0) {
          const float ratio = 1.2;  // hack
          v1 = s1->Goto(dt1, *s1);
          v1->s_ = s2->s_;
          s1->GotoWaypoint(0);
          //s1->GotoRest(dt1);
          //s1->GotoWaypoint(0);
          
          v2 = s2->Goto(dt1, *s2);
          v2->s_ = s1->s_;
          s2->GotoWaypoint(0);
          //s2->GotoRest(dt1);
          //s2->GotoWaypoint(0);
        }
        /* Different type of shape */
        else {
          s1->CreatePath(s2, s2->s_.x, dt1*3);
          s2->CreatePath(s1, s1->s_.x, dt1);
          s1->path_type_ = Vertex::PATH_SINGLE;
          s2->path_type_ = Vertex::PATH_SINGLE;
        }
        marq_type_ = MARQ_NONE;
        DeselectAll();
        break;
      case MARQ_TRI1:
        marq_x2_ = x;
        marq_y2_ = y;
        marq_type_ = MARQ_TRI2;
        marq_time2_ = time;
        break;
      case MARQ_TRI2:
        if (obj_selected_.size() < 3)
          return;
        dt1 = (marq_time2_ - marq_time_);
        dt2 = (time - marq_time2_);
        if (dt1 < 8.0) dt1 *= 2;
        if (dt2 < 8.0) dt2 *= 2;
        s1->ClearPath();
        s2->ClearPath();
        s3->ClearPath();
        v1 = s1->Goto(dt1, *s2);
        v1->r_ = 0;
        s1->GotoWaypoint(0);
        v2 = s2->Goto(dt2, *s3);
        v2->r_ = 0;
        s2->GotoWaypoint(0);
        v3 = s3->Goto((dt1 + dt2) / 2, *s1);
        v3->r_ = 0;
        s3->GotoWaypoint(0);
        DeselectAll();
        marq_type_ = MARQ_NONE;
        break;
      }
    }
    else {
      marq_type_ = MARQ_NONE;
      DeselectAll();
    }
  }
}

void
UIMap::MouseUp(int x, int y, int btn)
{
  if (btn == GLUT_RIGHT_BUTTON) {
    switch (rightstate_) {
    case STATE_PLACE:
    case STATE_THROW:
      obj_undercurs_ = 0;
      break;
    }
    rightstate_ = STATE_NONE;
    marq_type_ = MARQ_NONE;
    DeselectAll();
  }
}

void
UIMap::MouseMove(int x, int y)
{
  Mesh *s1;
  float selx, sely;

  /* Reset inactivity timer */
  inactivity_ = 0;

  switch (rightstate_) {
  case STATE_THROW:
    if (counter_ > 0.06) {
      rightstate_ = STATE_PLACE;
      s1 = CreateShape(curshape_, x, height_ - y);
      obj_undercurs_ = s1;
      counter_ = 0;
      period_ = 0;
      mouse_x_down_ = x;
      mouse_y_down_ = y;
    }
    break;
  case STATE_PLACE:
    if (effect_type_ == EFFECT_NONE && mouse_x_ != mouse_x_down_ && mouse_y_ != mouse_y_down_) {
      rightstate_ = STATE_THROW;
      counter_ = 0;
      period_ = 0;
    }
    break;
  }

  mouse_x_ = x;
  mouse_y_ = y;
	ComputeCoords(x, y, &selx, &sely);
  ml_pos_[0] = selx;
  ml_pos_[1] = sely;
  glLightfv(GL_LIGHT1, GL_POSITION, ml_pos_);
}

#if 0  
int
UIMap::HandleMouse(int x, int y, int btn, int state)
{
	Options &o = *(Options::instance());
  int result;
  float selx, sely;
  Mesh *s1, *s2, *s3, *s4;
  float time = o.GetTime();
  int idx;

  /*
   * Left button down
   */
  if (btn == GLUT_LEFT_BUTTON && state == GLUT_DOWN) {
		mouse_x_ = x;
		mouse_y_ = y;

		ComputeCoords(x, y, &selx, &sely);
    ml_pos_[0] = selx;
    ml_pos_[1] = sely;
    glLightfv(GL_LIGHT1, GL_POSITION, ml_pos_);

    // Select item under cursor
    if (FAIL(result = PickObject(x, y))) {
      return result;
    }

    if (obj_selected_.size() >= 1) s1 = obj_vec_[obj_selected_[0]];
    if (obj_selected_.size() >= 2) s2 = obj_vec_[obj_selected_[1]];
    if (obj_selected_.size() >= 3) s3 = obj_vec_[obj_selected_[2]];
    if (obj_selected_.size() >= 4) s4 = obj_vec_[obj_selected_[3]];

    /*
     * Object here?
     */
    if (obj_selected_.size() > 0) {
  	  glEnable(GL_LIGHT1);
      switch (marq_type_) {
      case MARQ_NONE:
        marq_x_ = x;
			  marq_y_ = y;
        marq_time_ = time;
        if (strcmp(s1->GetName(), "triangle") == 0) {
          marq_type_ = MARQ_TRI1;
        }
        else if (strcmp(s1->GetName(), "circle") == 0) {
          marq_type_ = MARQ_CIRC;
        }
        else {
          marq_type_ = MARQ_LINE;
        }
        break;
      case MARQ_LINE:
        /* Same type of shape? */
        if (strcmp(s1->GetName(), s2->GetName()) == 0) {
          s1->Merge(s2);
        }
        /* Different type of shape */
        else {
          vector3f m = (s1->p_ + s2->p_) / 2.0f;
          s1->Goto(time - marq_time_, m.x, m.y, m.z);
          s1->GotoRest(time - marq_time_);
          s2->Goto(time - marq_time_, m.x, m.y, m.z);
          s2->GotoRest(time - marq_time_);
        }
        printf("connection (time=%f)\n", (float)(time - marq_time_));
        marq_type_ = MARQ_NONE;
        glDisable(GL_LIGHT1);
        DeselectAll();
        break;
      case MARQ_CIRC:
        /* Same type of shape? */
        if (strcmp(s1->GetName(), s2->GetName()) == 0) {
          s1->Merge(s2);
        }
        /* Different type of shape */
        else {
          s1->CreatePath(s2, 10.0, (time - marq_time_) * 2);
        }
        marq_type_ = MARQ_NONE;
        glDisable(GL_LIGHT1);
        DeselectAll();
        break;
      case MARQ_TRI1:
        marq_x2_ = x;
        marq_y2_ = y;
        marq_type_ = MARQ_TRI2;
        marq_time2_ = time;
        break;
      case MARQ_TRI2:
        if (obj_selected_.size() > 2) {
          float dt1 = marq_time2_ - marq_time_;
          float dt2 = time - marq_time2_;
          s1 = obj_vec_[obj_selected_[0]];
          s2 = obj_vec_[obj_selected_[1]];
          s3 = obj_vec_[obj_selected_[2]];
          s1->ClearPath();
          s2->ClearPath();
          s3->ClearPath();
          s1->Goto(dt1, s2->p_);
          s1->Goto(dt2, s3->p_);
          s1->GotoRest((dt1 + dt2) / 2);
          s2->Goto(dt2, s3->p_);
          s2->Goto((dt1 + dt2) / 2, s1->p_);
          s2->GotoRest(dt1);
          s3->Goto((dt1 + dt2) / 2, s1->p_);
          s3->Goto(dt1, s2->p_);
          s3->GotoRest(dt2);
          DeselectAll();
        }
        marq_type_ = MARQ_NONE;
        glDisable(GL_LIGHT1);
        break;
      }
    }
    else {
      marq_type_ = MARQ_NONE;
      glDisable(GL_LIGHT1);
      DeselectAll();
    }

    return ERR_NEEDREDRAW;
  }

  /*
   * Left button up
   */
  else if (btn == GLUT_LEFT_BUTTON && state == GLUT_UP) {
	  
  }
  
  /*
   * Mouse move
   */
  else if (btn == -1 && state == -1) {
    mouse_x_ = x;
    mouse_y_ = y;

		ComputeCoords(x, y, &selx, &sely);
    ml_pos_[0] = selx;
    ml_pos_[1] = sely;
    glLightfv(GL_LIGHT1, GL_POSITION, ml_pos_);
   
  }

  return ERR_OK;
}
#endif

int
UIMap::HandleKey(unsigned char key, int state)
{
	Options &o = *(Options::instance());
  int i;

  int cellwidth = width_ / rgnmap_.width;
  int cellheight = height_ / rgnmap_.height;

  switch (key) {
  case 27:  /* ESC */
    marq_type_ = MARQ_NONE;
    /* Delete all objects */
    for (i = 0; i < obj_vec_.size(); i++) {
      delete obj_vec_[i];
    }
    obj_vec_.clear();
    obj_selected_.clear();
    obj_undercurs_ = 0;
    break;
  case 'a':
    camera_x_ -= cellwidth;
    break;
  case 'd':
    camera_x_ += cellwidth;
    break;
  case 'w':
    camera_y_ += cellheight;
    break;
  case 's':
    camera_y_ -= cellheight;
    break;
  case '-':
    camera_z_ *= 1.1;
    break;
  case '=':
    camera_z_ *= 0.9;
    break;
  case 'r':
    if (effect_type_ == EFFECT_NONE) {
      ShootUp();
    }
    break;
  case 'f':
    if (effect_type_ == EFFECT_NONE) {
      ShootDown();
    }
    break;
  }

  printf("%d\n", key);

  return ERR_NEEDREDRAW;
}

void
UIMap::SetMouseScale(void)
{
	float x1, y1;
	float x2, y2;
	
	ComputeCoords(0, 0, &x1, &y1);
	ComputeCoords(1, 1, &x2, &y2);

	mouse_scalex_ = (x2 - x1);
	mouse_scaley_ = (y1 - y2);
}

void
UIMap::DeselectAll(void)
{
	for (int i = 0; i < obj_selected_.size(); i++) {
		obj_vec_[ obj_selected_[i] ]->SetSelected(false);
	}
	obj_selected_.clear();
}

void 
UIMap::ApplyTextureToSelection(Texture *texobj)
{
	for (int i = 0; i < obj_selected_.size(); i++) {
		//obj_vec_[ obj_selected_[i] ]->SetTexture(texobj);
	}
}

void
UIMap::DoConnection(Mesh *s1, Mesh *s2)
{
  vector3f m = (s1->p_ + s2->p_) / 2.0f;
  s1->Goto(2.0f, m.x, m.y, m.z);
  s2->Goto(2.0f, m.x, m.y, m.z);
}

void
UIMap::SetAverageColor(int left, int top, int right, int bottom, Mesh *obj)
{
  /* Average the specified rectangular region of the background image */
  int idx;
  const int size = (right - left) * (bottom - top) * 255;
  int r = 0, g = 0, b = 0;
  for (int j = top; j < bottom; j++) {
    idx = (j * img_width_ + left) * 4;  /* HACK HACK HACK */
    for (int i = left; i < right; i++) {
      r += img_[idx++];
      g += img_[idx++];
      b += img_[idx++];
      idx++;  // skip alpha
    }
  }
  obj->SetColor((float)r / size, (float)g / size, (float)b / size);
}

Mesh* 
UIMap::CreateShape(int type, int x, int y)
{
  Mesh *shape;
  int cellwidth = width_ / rgnmap_.width;
  int cellheight = height_ / rgnmap_.height;
  int cellwidth2 = cellwidth / 2;
  int cellheight2 = cellheight / 2;
  //int cellx = ((float)img_width_ / (float)width_) * (float)x / cellwidth;
  //int celly = ((float)img_height_ / (float)height_) * (float)y / cellheight;
  int cellx = (float)x / cellwidth;
  int celly = (float)y / cellheight;
  REGION_INFO &ri = rgnmap_.map[(int)rgnmap_.mask[celly * rgnmap_.width + cellx]];
  switch (type) {
  case 0:
	  shape = new Parallelogram(x, y, rand() % 5, cellheight - 3);
    shape->SetName("parallelogram");
    obj_vec_.push_back(shape);
	  break;
  case 1:
	  shape = new Circle(x, y, rand() % 5, cellheight2 - 1);
    shape->SetName("circle");
    obj_vec_.push_back(shape);
	  break;
  case 2:
	  shape = new Square(x, y, rand() % 5, cellheight - 5);
    shape->SetName("square");
    obj_vec_.push_back(shape);
	  break;
  case 3:
	  shape = new Triangle(x, y, rand() % 5, cellwidth - 3, cellheight - 1);
    shape->SetName("triangle");
    obj_vec_.push_back(shape);
    break;
  default:
	  shape = new Square(x, y, rand() % 5, 4);
    shape->SetName("default");
    obj_vec_.push_back(shape);
    break;
  }
  shape->SetColor(ri.r / 255.0, ri.g / 255.0, ri.b / 255.0);
  
#if 0
  /* 
   * Set the color based on the region we're in.  The shape names below 
   * are completely misleading and should be changed.
   */
  switch(rgnmap_.map[].region) {
  case PARALLELOGRAM:
    shape->SetColor(0.8f, 0.6f, 0.2f);  /* parallelogram */
	  break;
  case CIRCLE:
    shape->SetColor(0.0, 0.4f, 0.8f);   /* circle */
	  break;
  case SQUARE:
    shape->SetColor(1.0f, 1.0f, 0.4f);  /* square */
	  break;
  case TRIANGLE:
    shape->SetColor(0.7f, 0.1f, 0.3f);  /* triangle */
    break;
  }
#endif

  return shape;
}

/**********************************************************
 * 
 * Special Effects
 *
 * Applies to geometry.
 *
 **********************************************************/

void
UIMap::ShootUp(void)
{
  effect_type_ = EFFECT_SHOOTUP;
  period_ = 3.0;
  counter_ = 0;
  for (int i = 0; i < obj_vec_.size(); i++) {
    obj_vec_[i]->ClearPath();
    obj_vec_[i]->Goto(3.0, rand() % 1280 - 320, rand() % 960 - 240, camera_z_ * 1.1);
  }
  Options &o = *Options::instance();
  o.light_position_[2] += 400;
	glLightfv(GL_LIGHT0, GL_POSITION, o.light_position_);
}

void
UIMap::ShootDown(void)
{
  effect_type_ = EFFECT_SHOOTDOWN;
  period_ = 3.0;
  counter_ = 0;
  for (int i = 0; i < obj_vec_.size(); i++) {
    obj_vec_[i]->ClearPath();
    obj_vec_[i]->GotoRest(3.0);
  }
}



/**********************************************************
 * 
 * PickObject()
 *
 * Selects an object at (x, y)
 * From Ch. 13 (p. 549) in red book.
 *
 **********************************************************/

int 
UIMap::PickObject(int x, int y)
{
	int selectbufsize = 512;
	unsigned int selectbuf[512];
	int hits;
	int viewport[4];

	//glViewport(32, 0, 602, 436);


	// Get viewport dimensions
	glGetIntegerv(GL_VIEWPORT, viewport);
	int x1 = viewport[0];
	int y1 = viewport[1];
	int width = viewport[2];
	int height = viewport[3];

	glSelectBuffer(selectbufsize, selectbuf);
	glRenderMode(GL_SELECT);

	glInitNames();
	glPushName(0);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	// Create a 2x2 picking region near (x, y)
	gluPickMatrix((GLdouble)x + x1, (GLdouble)(height - y), 2.0, 2.0, viewport);

	gluPerspective(fov_, (GLfloat)(width) / (GLfloat)(height), near_, far_);
	glTranslatef(-camera_x_, -camera_y_, -camera_z_);

	if (FAIL(Draw(GL_SELECT))) {
		return ERR_FAIL;
	}

	glPopMatrix();
	glFlush();

	hits = glRenderMode(GL_RENDER);
  /* 
   * Groundplane is not rendered for a GL_SELECT, so if hits > 0
   * then we have clicked on an object.  ie, groundplance does not
   * count.
   */
	if (hits > 0) {
    //DeselectAll(); /* one thing at a time */
		if (FAIL(ProcessHits(hits, selectbuf))) {
			return ERR_FAIL;
		}
	}
	else {
		DeselectAll();
	}

	return ERR_OK;

}

/**********************************************************
 * 
 * SelectObjects(int x1, int y1, int x2, int y2)
 *
 * Selects all objects inside screen coords (x1, y1) to (x2, y2)
 * From Ch. 13 (p. 539) in red book.
 *
 **********************************************************/

int 
UIMap::SelectObjects(int x1, int y1, int x2, int y2)
{
	int selectbufsize = 512;
	unsigned int selectbuf[512];
	int hits;
	int viewport[4];

	//glViewport(32, 0, 602, 436);


	// Get viewport dimensions
	glGetIntegerv(GL_VIEWPORT, viewport);
	int x = viewport[0];
	int y = viewport[1];
	int width = viewport[2];
	int height = viewport[3];

	glSelectBuffer(selectbufsize, selectbuf);
	glRenderMode(GL_SELECT);

	glInitNames();
	glPushName(0);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	// Create a big picking region near (x, y)
	double centerx = (x1 + x2) / 2;
	double centery = (y1 + y2) / 2;
	gluPickMatrix((GLdouble)centerx + x, (GLdouble)(height - centery), ABS(x2 - x1), ABS(y2 - y1), viewport);

	gluPerspective(fov_, (GLfloat)(width) / (GLfloat)(height), near_, far_);
	glTranslatef(-camera_x_, -camera_y_, -camera_z_);

	if (FAIL(Draw(GL_SELECT))) {
		return ERR_FAIL;
	}

	glPopMatrix();
	glFlush();

	hits = glRenderMode(GL_RENDER);
	if (hits > 0) {
		if (FAIL(ProcessHits(hits, selectbuf))) {
			return ERR_FAIL;
		}
	}
	else {
		DeselectAll();
	}

	return ERR_OK;
}


int
UIMap::ProcessHits(int hits, unsigned int buffer[]) 
{
	int i, j;
	int names;
	float z1, z2;
	int objid;
	unsigned int *ptr;
  int idx;

	ptr = (GLuint*)buffer;
	for (i = 0; i < hits; i++) {
		names = *ptr; ptr++;
		z1 = (float)*ptr/0x7fffffff; ptr++;
		z2 = (float)*ptr/0x7fffffff; ptr++;
		for (j = 0; j < names; j++) {
			objid = *ptr; 
      if (objid >= obj_vec_.size()) {
        /* Not an object.  Maybe a button? */
        idx = objid - obj_vec_.size();
        if (idx < buttons_.size()) {
          if (!buttons_[idx]->IsSelected()) {
            buttons_selected_.push_back(idx);
            buttons_[idx]->SetSelected(true);
          }
        }
      }
			else if (!obj_vec_[objid]->IsSelected()) {
				obj_selected_.push_back(objid);
				obj_vec_[objid]->SetSelected(true);
			}
			ptr++;
		}
	}
	return ERR_OK;
}


/*
** This is a screwball function.  What it does is the following:
** Given screen x and y coordinates, compute the corresponding object space 
**   x and y coordinates given that the object space z is 0.
*/
int
UIMap::ComputeCoords(int mousex, int mousey, GLfloat * selx, GLfloat * sely)
{
	GLfloat modelMatrix[16];
	GLfloat projMatrix[16];
	GLfloat finalMatrix[16];
	GLfloat in[4];
	GLfloat a, b, c, d;
	GLfloat top, bot;
	GLfloat z;
	GLfloat w;
	GLfloat height;
	GLint viewport[4];

	// Get viewport dimensions
	glGetIntegerv(GL_VIEWPORT, viewport);
	
	//height = zsize[piece] - 0.1 + OFFSETZ;
	height = 0;
	
	glGetFloatv(GL_PROJECTION_MATRIX, projMatrix);
	glGetFloatv(GL_MODELVIEW_MATRIX, modelMatrix);
	multMatrices(modelMatrix, projMatrix, finalMatrix);
	if (!invertMatrix(finalMatrix, finalMatrix))
		return 0;
	
	//in[0] = (2.0 * (mousex - viewport[0]) / viewport[2]) - 1;
	//in[1] = (2.0 * ((H - mousey) - viewport[1]) / viewport[3]) - 1;
	in[0] = (2.0 * (mousex) / viewport[2]) - 1;
	in[1] = (2.0 * ((viewport[3] - mousey)) / viewport[3]) - 1;
	
	a = in[0] * finalMatrix[0 * 4 + 2] +
		in[1] * finalMatrix[1 * 4 + 2] +
		finalMatrix[3 * 4 + 2];
	b = finalMatrix[2 * 4 + 2];
	c = in[0] * finalMatrix[0 * 4 + 3] +
		in[1] * finalMatrix[1 * 4 + 3] +
		finalMatrix[3 * 4 + 3];
	d = finalMatrix[2 * 4 + 3];
	
	/* 
	** Ok, now we need to solve for z: 
	**   (a + b z) / (c + d z) = height. 
	** ("height" is the height in object space we want to solve z for) 
	** 
	** ==>  a + b z = height c + height d z 
	**      bz - height d z = height c - a 
	** z = (height c - a) / (b - height d) 
	*/

	top = height * c - a;
	bot = b - height * d;
	if (bot == 0.0)
		return 0;
	
	z = top / bot;
	
	/* 
	** Ok, no problem. 
	** Now we solve for x and y.  We know that w = c + d z, so we compute it. 
	*/
	
	w = c + d * z;
	
	/* 
	** Now for x and y: 
	*/
	
	*selx = (in[0] * finalMatrix[0 * 4 + 0] +
		in[1] * finalMatrix[1 * 4 + 0] +
		z * finalMatrix[2 * 4 + 0] +
		finalMatrix[3 * 4 + 0]) / w;
	*sely = (in[0] * finalMatrix[0 * 4 + 1] +
		in[1] * finalMatrix[1 * 4 + 1] +
		z * finalMatrix[2 * 4 + 1] +
		finalMatrix[3 * 4 + 1]) / w;
	return 1;
}


